<?php
/**
 * Instalador Profissional - Portal Bahia
 * Desenvolvido por Robson Rost / Grupo Website VIX
 */

session_start();

// Configurações
define('INSTALL_VERSION', '1.0.0');
define('REQUIRED_PHP_VERSION', '8.1');
define('REQUIRED_EXTENSIONS', ['pdo', 'pdo_mysql', 'mbstring', 'openssl', 'curl', 'fileinfo', 'json', 'xml']);

// Verificar se já está instalado
if (file_exists(__DIR__ . '/../.env') && file_exists(__DIR__ . '/../storage/installed')) {
    header('Location: /admin/login');
    exit;
}

$step = $_GET['step'] ?? 'welcome';
$errors = [];
$messages = [];

// Processar formulários
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    switch ($_POST['step']) {
        case 'requirements':
            $step = 'database';
            break;
            
        case 'database':
            // Validar e testar conexão
            $dbHost = $_POST['db_host'] ?? 'localhost';
            $dbPort = $_POST['db_port'] ?? '3306';
            $dbName = $_POST['db_name'] ?? '';
            $dbUser = $_POST['db_user'] ?? '';
            $dbPass = $_POST['db_pass'] ?? '';
            
            try {
                $dsn = "mysql:host={$dbHost};port={$dbPort};charset=utf8mb4";
                $pdo = new PDO($dsn, $dbUser, $dbPass);
                $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                
                // Criar banco se não existir
                $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$dbName}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
                
                // Salvar dados
                $_SESSION['install']['database'] = [
                    'host' => $dbHost,
                    'port' => $dbPort,
                    'name' => $dbName,
                    'user' => $dbUser,
                    'pass' => $dbPass
                ];
                
                $step = 'license';
                $messages[] = 'Conexão com banco de dados estabelecida com sucesso!';
            } catch (Exception $e) {
                $errors[] = 'Erro ao conectar ao banco de dados: ' . $e->getMessage();
            }
            break;
            
        case 'license':
            $licenseKey = trim($_POST['license_key'] ?? '');
            $masterUrl = trim($_POST['master_url'] ?? 'https://master.canal33.com.br');
            
            if (empty($licenseKey)) {
                $errors[] = 'License Key é obrigatória';
            } else {
                // Obter IP e domínio para registrar instalação
                $ip = $_SERVER['SERVER_ADDR'] ?? $_SERVER['REMOTE_ADDR'] ?? null;
                $domain = $_SERVER['HTTP_HOST'] ?? null;
                
                // Validar licença com Master Panel (enviando IP e domínio)
                // Tentar primeiro o endpoint direto (sem mod_rewrite)
                $validationUrl = rtrim($masterUrl, '/') . '/api/v1/validate-license.php?license_key=' . urlencode($licenseKey);
                
                // Adicionar IP e domínio se disponíveis
                if ($ip) {
                    $validationUrl .= '&ip=' . urlencode($ip);
                }
                if ($domain) {
                    $validationUrl .= '&domain=' . urlencode($domain);
                }
                
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $validationUrl);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_setopt($ch, CURLOPT_TIMEOUT, 15);
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
                
                $response = curl_exec($ch);
                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                $curlError = curl_error($ch);
                curl_close($ch);
                
                if ($curlError) {
                    $errors[] = 'Erro de conexão: ' . $curlError . '. Verifique a URL do Master Panel e sua conexão com a internet.';
                } elseif ($httpCode === 200) {
                    $result = json_decode($response, true);
                    if ($result && isset($result['success']) && $result['success']) {
                        // Licença válida
                        $_SESSION['install']['license'] = [
                            'key' => $licenseKey,
                            'master_url' => $masterUrl,
                            'client' => $result['license']['client'] ?? null
                        ];
                        
                        $step = 'admin';
                        $messages[] = 'Licença validada com sucesso! IP e domínio registrados.';
                    } else {
                        $errorMsg = $result['message'] ?? 'Licença inválida ou expirada';
                        $errors[] = $errorMsg . '. Verifique sua License Key no Master Panel.';
                    }
                } else {
                    $result = json_decode($response, true);
                    $errorMsg = $result['message'] ?? 'Erro ao validar licença';
                    
                    if ($httpCode === 403) {
                        $errors[] = $errorMsg . '. Verifique se a licença está ativa no Master Panel.';
                    } elseif ($httpCode === 404) {
                        $errors[] = 'Endpoint não encontrado. Verifique se a URL do Master Panel está correta: ' . $masterUrl;
                    } elseif ($httpCode >= 500) {
                        $errors[] = 'Erro no servidor Master Panel. Tente novamente mais tarde.';
                    } else {
                        $errors[] = $errorMsg . ' (Status: ' . $httpCode . '). Verifique a URL do Master Panel e sua conexão com a internet.';
                    }
                }
            }
            break;
            
        case 'admin':
            $adminName = trim($_POST['admin_name'] ?? '');
            $adminEmail = trim($_POST['admin_email'] ?? '');
            $adminPassword = $_POST['admin_password'] ?? '';
            $adminPasswordConfirm = $_POST['admin_password_confirm'] ?? '';
            $siteName = trim($_POST['site_name'] ?? 'Portal Bahia');
            
            // Validações
            if (empty($adminName)) {
                $errors[] = 'Nome do administrador é obrigatório';
            }
            if (empty($adminEmail) || !filter_var($adminEmail, FILTER_VALIDATE_EMAIL)) {
                $errors[] = 'Email válido é obrigatório';
            }
            if (strlen($adminPassword) < 8) {
                $errors[] = 'Senha deve ter no mínimo 8 caracteres';
            }
            if ($adminPassword !== $adminPasswordConfirm) {
                $errors[] = 'Senhas não coincidem';
            }
            
            if (empty($errors)) {
                $_SESSION['install']['admin'] = [
                    'name' => $adminName,
                    'email' => $adminEmail,
                    'password' => password_hash($adminPassword, PASSWORD_BCRYPT)
                ];
                $_SESSION['install']['site'] = [
                    'name' => $siteName
                ];
                
                $step = 'install';
                // Iniciar instalação
                install();
            }
            break;
    }
}

/**
 * Executar instalação
 */
function install() {
    try {
        $db = $_SESSION['install']['database'];
        $license = $_SESSION['install']['license'];
        $admin = $_SESSION['install']['admin'];
        $site = $_SESSION['install']['site'];
        
        // Conectar ao banco
        $dsn = "mysql:host={$db['host']};port={$db['port']};dbname={$db['name']};charset=utf8mb4";
        $pdo = new PDO($dsn, $db['user'], $db['pass']);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // Importar SQL
        $sqlFile = __DIR__ . '/../database/schema.sql';
        if (file_exists($sqlFile)) {
            $sql = file_get_contents($sqlFile);
            $pdo->exec($sql);
        }
        
        // Criar administrador
        $stmt = $pdo->prepare("INSERT INTO users (name, email, password, role, status, created_at) VALUES (?, ?, ?, 'admin', 'active', NOW())");
        $stmt->execute([$admin['name'], $admin['email'], $admin['password']]);
        
        // Criar arquivo .env
        $envContent = generateEnvFile($db, $license, $site);
        file_put_contents(__DIR__ . '/../.env', $envContent);
        
        // Criar arquivo de instalação
        file_put_contents(__DIR__ . '/../storage/installed', date('Y-m-d H:i:s'));
        
        // Criar diretórios necessários
        $dirs = [
            __DIR__ . '/../storage/app/public/uploads',
            __DIR__ . '/../storage/logs',
            __DIR__ . '/../storage/framework/cache',
            __DIR__ . '/../storage/framework/sessions',
            __DIR__ . '/../storage/framework/views',
        ];
        
        foreach ($dirs as $dir) {
            if (!is_dir($dir)) {
                mkdir($dir, 0755, true);
            }
        }
        
        $_SESSION['install']['success'] = true;
        $step = 'complete';
        
    } catch (Exception $e) {
        $errors[] = 'Erro durante instalação: ' . $e->getMessage();
        error_log("Install error: " . $e->getMessage());
    }
}

function generateEnvFile($db, $license, $site) {
    return "APP_NAME=\"{$site['name']}\"
APP_ENV=production
APP_KEY=
APP_DEBUG=false
APP_URL=http://localhost

LOG_CHANNEL=stack
LOG_LEVEL=debug

DB_CONNECTION=mysql
DB_HOST={$db['host']}
DB_PORT={$db['port']}
DB_DATABASE={$db['name']}
DB_USERNAME={$db['user']}
DB_PASSWORD={$db['pass']}

MASTER_PANEL_URL={$license['master_url']}
MASTER_PANEL_LICENSE_KEY={$license['key']}

BROADCAST_DRIVER=log
CACHE_DRIVER=file
FILESYSTEM_DISK=local
QUEUE_CONNECTION=sync
SESSION_DRIVER=file
SESSION_LIFETIME=120

MEMCACHED_HOST=127.0.0.1

REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379

MAIL_MAILER=smtp
MAIL_HOST=mailhog
MAIL_PORT=1025
MAIL_USERNAME=null
MAIL_PASSWORD=null
MAIL_ENCRYPTION=null
MAIL_FROM_ADDRESS=\"noreply@{$site['name']}\"
MAIL_FROM_NAME=\"{$site['name']}\"

AWS_ACCESS_KEY_ID=
AWS_SECRET_ACCESS_KEY=
AWS_DEFAULT_REGION=us-east-1
AWS_BUCKET=
AWS_USE_PATH_STYLE_ENDPOINT=false

VITE_APP_NAME=\"{$site['name']}\"
";
}

// Verificar requisitos
function checkRequirements() {
    $checks = [
        'php_version' => version_compare(PHP_VERSION, REQUIRED_PHP_VERSION, '>='),
        'extensions' => [],
        'writable_dirs' => []
    ];
    
    foreach (REQUIRED_EXTENSIONS as $ext) {
        $checks['extensions'][$ext] = extension_loaded($ext);
    }
    
    $writableDirs = [
        __DIR__ . '/../storage',
        __DIR__ . '/../bootstrap/cache'
    ];
    
    foreach ($writableDirs as $dir) {
        $checks['writable_dirs'][$dir] = is_writable($dir) || (!file_exists($dir) && is_writable(dirname($dir)));
    }
    
    return $checks;
}

$requirements = checkRequirements();
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instalação - Portal Bahia</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 2rem;
        }
        .container {
            background: white;
            border-radius: 10px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            max-width: 800px;
            width: 100%;
            padding: 2rem;
        }
        h1 { color: #333; margin-bottom: 0.5rem; }
        .subtitle { color: #666; margin-bottom: 2rem; }
        .step-indicator {
            display: flex;
            justify-content: space-between;
            margin-bottom: 2rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid #e5e7eb;
        }
        .step {
            flex: 1;
            text-align: center;
            padding: 0.5rem;
            position: relative;
        }
        .step.active { color: #667eea; font-weight: bold; }
        .step.completed { color: #10b981; }
        .step::after {
            content: '';
            position: absolute;
            bottom: -1rem;
            left: 50%;
            transform: translateX(-50%);
            width: 100%;
            height: 2px;
            background: #e5e7eb;
        }
        .step:last-child::after { display: none; }
        .step.active::after { background: #667eea; }
        .step.completed::after { background: #10b981; }
        .form-group {
            margin-bottom: 1.5rem;
        }
        label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 500;
            color: #333;
        }
        input[type="text"],
        input[type="email"],
        input[type="password"],
        input[type="number"],
        select,
        textarea {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #ddd;
            border-radius: 5px;
            font-size: 1rem;
        }
        .btn {
            background: #667eea;
            color: white;
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 5px;
            font-size: 1rem;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
        }
        .btn:hover { background: #5568d3; }
        .btn-success { background: #10b981; }
        .btn-success:hover { background: #059669; }
        .alert {
            padding: 1rem;
            border-radius: 5px;
            margin-bottom: 1rem;
        }
        .alert-error {
            background: #fee2e2;
            color: #991b1b;
            border: 1px solid #ef4444;
        }
        .alert-success {
            background: #d1fae5;
            color: #065f46;
            border: 1px solid #10b981;
        }
        .alert-info {
            background: #dbeafe;
            color: #1e40af;
            border: 1px solid #3b82f6;
        }
        .check-list {
            list-style: none;
            margin: 1rem 0;
        }
        .check-list li {
            padding: 0.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        .check-list li::before {
            content: '✓';
            color: #10b981;
            font-weight: bold;
            font-size: 1.2rem;
        }
        .check-list li.error::before {
            content: '✗';
            color: #ef4444;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Instalação do Portal Bahia</h1>
        <p class="subtitle">Versão <?php echo INSTALL_VERSION; ?> - Desenvolvido por Robson Rost / Grupo Website VIX</p>
        
        <div class="step-indicator">
            <div class="step <?php echo $step === 'welcome' || $step === 'requirements' ? 'active' : ($step === 'database' || $step === 'license' || $step === 'admin' || $step === 'install' || $step === 'complete' ? 'completed' : ''); ?>">1. Requisitos</div>
            <div class="step <?php echo $step === 'database' ? 'active' : ($step === 'license' || $step === 'admin' || $step === 'install' || $step === 'complete' ? 'completed' : ''); ?>">2. Banco</div>
            <div class="step <?php echo $step === 'license' ? 'active' : ($step === 'admin' || $step === 'install' || $step === 'complete' ? 'completed' : ''); ?>">3. Licença</div>
            <div class="step <?php echo $step === 'admin' || $step === 'install' ? 'active' : ($step === 'complete' ? 'completed' : ''); ?>">4. Admin</div>
            <div class="step <?php echo $step === 'complete' ? 'active' : ''; ?>">5. Concluído</div>
        </div>
        
        <?php if (!empty($errors)): ?>
            <?php foreach ($errors as $error): ?>
            <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
            <?php endforeach; ?>
        <?php endif; ?>
        
        <?php if (!empty($messages)): ?>
            <?php foreach ($messages as $msg): ?>
            <div class="alert alert-success"><?php echo htmlspecialchars($msg); ?></div>
            <?php endforeach; ?>
        <?php endif; ?>
        
        <?php if ($step === 'welcome'): ?>
            <div class="alert alert-info">
                <h3>Bem-vindo ao Instalador do Portal Bahia!</h3>
                <p>Este instalador irá configurar seu portal de notícias em alguns passos simples.</p>
            </div>
            
            <ul class="check-list">
                <li>PHP <?php echo REQUIRED_PHP_VERSION; ?> ou superior</li>
                <li>Extensões PHP necessárias</li>
                <li>Banco de dados MySQL</li>
                <li>License Key do Master Panel</li>
            </ul>
            
            <form method="POST">
                <input type="hidden" name="step" value="requirements">
                <button type="submit" class="btn">Verificar Requisitos</button>
            </form>
            
        <?php elseif ($step === 'requirements'): ?>
            <h2>Verificação de Requisitos</h2>
            
            <h3>Versão do PHP</h3>
            <ul class="check-list">
                <li class="<?php echo $requirements['php_version'] ? '' : 'error'; ?>">
                    PHP <?php echo PHP_VERSION; ?> <?php echo $requirements['php_version'] ? '✓ Atende' : '✗ Não atende'; ?>
                </li>
            </ul>
            
            <h3>Extensões PHP</h3>
            <ul class="check-list">
                <?php foreach ($requirements['extensions'] as $ext => $loaded): ?>
                <li class="<?php echo $loaded ? '' : 'error'; ?>">
                    <?php echo $ext; ?> <?php echo $loaded ? '✓ Instalada' : '✗ Não instalada'; ?>
                </li>
                <?php endforeach; ?>
            </ul>
            
            <h3>Permissões de Diretórios</h3>
            <ul class="check-list">
                <?php foreach ($requirements['writable_dirs'] as $dir => $writable): ?>
                <li class="<?php echo $writable ? '' : 'error'; ?>">
                    <?php echo basename($dir); ?> <?php echo $writable ? '✓ Gravável' : '✗ Não gravável'; ?>
                </li>
                <?php endforeach; ?>
            </ul>
            
            <?php
            $allOk = $requirements['php_version'] && 
                     !in_array(false, $requirements['extensions']) && 
                     !in_array(false, $requirements['writable_dirs']);
            ?>
            
            <?php if ($allOk): ?>
            <form method="POST" action="?step=database">
                <input type="hidden" name="step" value="database">
                <button type="submit" class="btn btn-success">Continuar →</button>
            </form>
            <?php else: ?>
            <div class="alert alert-error">
                Corrija os erros acima antes de continuar.
            </div>
            <?php endif; ?>
            
        <?php elseif ($step === 'database'): ?>
            <h2>Configuração do Banco de Dados</h2>
            <form method="POST">
                <input type="hidden" name="step" value="database">
                
                <div class="form-group">
                    <label>Host do Banco de Dados *</label>
                    <input type="text" name="db_host" value="localhost" required>
                </div>
                
                <div class="form-group">
                    <label>Porta</label>
                    <input type="number" name="db_port" value="3306" required>
                </div>
                
                <div class="form-group">
                    <label>Nome do Banco de Dados *</label>
                    <input type="text" name="db_name" value="canal33portal_portalnoticia" required>
                    <small>O banco será criado automaticamente se não existir</small>
                </div>
                
                <div class="form-group">
                    <label>Usuário *</label>
                    <input type="text" name="db_user" value="canal33portal_portalnoticia" required>
                </div>
                
                <div class="form-group">
                    <label>Senha *</label>
                    <input type="password" name="db_pass" value="VtlOq2WsS9vjuE1A" required>
                </div>
                
                <button type="submit" class="btn">Testar Conexão e Continuar →</button>
            </form>
            
        <?php elseif ($step === 'license'): ?>
            <h2>Validação de Licença</h2>
            <p>Você precisa de uma License Key válida do Master Panel para instalar o portal.</p>
            
            <form method="POST">
                <input type="hidden" name="step" value="license">
                
                <div class="form-group">
                    <label>URL do Master Panel *</label>
                    <input type="url" name="master_url" value="https://master.canal33.com.br" required>
                </div>
                
                <div class="form-group">
                    <label>License Key *</label>
                    <input type="text" name="license_key" required placeholder="PVX-XXXXXXXX-XXXXXXXX">
                    <small>Obtenha sua License Key no painel master</small>
                </div>
                
                <button type="submit" class="btn">Validar Licença e Continuar →</button>
            </form>
            
        <?php elseif ($step === 'admin'): ?>
            <h2>Configuração do Administrador</h2>
            
            <form method="POST">
                <input type="hidden" name="step" value="admin">
                
                <div class="form-group">
                    <label>Nome do Site</label>
                    <input type="text" name="site_name" value="Portal Bahia" required>
                </div>
                
                <div class="form-group">
                    <label>Nome do Administrador *</label>
                    <input type="text" name="admin_name" required>
                </div>
                
                <div class="form-group">
                    <label>Email do Administrador *</label>
                    <input type="email" name="admin_email" required>
                </div>
                
                <div class="form-group">
                    <label>Senha *</label>
                    <input type="password" name="admin_password" required minlength="8">
                    <small>Mínimo de 8 caracteres</small>
                </div>
                
                <div class="form-group">
                    <label>Confirmar Senha *</label>
                    <input type="password" name="admin_password_confirm" required minlength="8">
                </div>
                
                <button type="submit" class="btn">Instalar Portal →</button>
            </form>
            
        <?php elseif ($step === 'install'): ?>
            <h2>Instalando...</h2>
            <p>Aguarde enquanto o portal está sendo instalado. Isso pode levar alguns minutos.</p>
            <div style="text-align: center; margin: 2rem 0;">
                <div style="display: inline-block; width: 50px; height: 50px; border: 5px solid #f3f3f3; border-top: 5px solid #667eea; border-radius: 50%; animation: spin 1s linear infinite;"></div>
            </div>
            <script>
                // Redirecionar após instalação
                setTimeout(function() {
                    window.location.href = '?step=complete';
                }, 3000);
            </script>
            <style>
                @keyframes spin {
                    0% { transform: rotate(0deg); }
                    100% { transform: rotate(360deg); }
                }
            </style>
            
        <?php elseif ($step === 'complete' && isset($_SESSION['install']['success'])): ?>
            <h2>✅ Instalação Concluída!</h2>
            <div class="alert alert-success">
                <p>O Portal Bahia foi instalado com sucesso!</p>
            </div>
            
            <div style="margin: 2rem 0;">
                <h3>Próximos Passos:</h3>
                <ol style="margin-left: 1.5rem; margin-top: 1rem;">
                    <li>Acesse o painel administrativo</li>
                    <li>Configure suas preferências</li>
                    <li>Comece a criar conteúdo!</li>
                </ol>
            </div>
            
            <a href="/admin/login" class="btn btn-success">Ir para o Painel Administrativo →</a>
            
            <?php
            // Limpar sessão de instalação
            unset($_SESSION['install']);
            ?>
            
        <?php endif; ?>
    </div>
</body>
</html>

