<?php

namespace App\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class MasterPanelService
{
    protected $baseUrl;
    protected $licenseKey;
    protected $apiKey;
    protected $apiSecret;

    public function __construct()
    {
        $this->baseUrl = config('master-panel.url', 'https://master.canal33.com.br');
        $this->licenseKey = config('master-panel.license_key');
        $this->apiKey = config('master-panel.api_key');
        $this->apiSecret = config('master-panel.api_secret');
    }

    /**
     * Validar licença
     */
    public function validateLicense($ip = null, $domain = null)
    {
        // Se não há chave de licença, retornar erro
        if (empty($this->licenseKey)) {
            return [
                'success' => false,
                'message' => 'Chave de licença não configurada. Configure MASTER_PANEL_LICENSE_KEY no arquivo .env'
            ];
        }

        // Se não há URL do Master Panel, retornar erro
        if (empty($this->baseUrl)) {
            return [
                'success' => false,
                'message' => 'URL do Master Panel não configurada. Configure MASTER_PANEL_URL no arquivo .env'
            ];
        }

        $cacheKey = 'master_license_validation_' . md5($this->licenseKey . $ip . $domain);
        
        if (config('master-panel.cache_license_validation', true)) {
            $cached = Cache::get($cacheKey);
            if ($cached && isset($cached['success'])) {
                return $cached;
            }
        }

        try {
            $url = rtrim($this->baseUrl, '/') . '/api/v1/validate-license';
            $params = ['license_key' => $this->licenseKey];
            
            if ($ip) {
                $params['ip'] = $ip;
            }
            if ($domain) {
                $params['domain'] = $domain;
            }

            $timeout = config('master-panel.api_timeout', 10);
            
            // Log da tentativa de validação
            Log::info('Validating license', [
                'url' => $url,
                'license_key' => substr($this->licenseKey, 0, 10) . '...',
                'ip' => $ip,
                'domain' => $domain
            ]);
            
            $response = Http::timeout($timeout)
                ->retry(2, 100) // Tentar 2 vezes com delay de 100ms
                ->get($url, $params);
            
            // Log da resposta
            Log::info('License validation response', [
                'status' => $response->status(),
                'body' => $response->json()
            ]);

            if ($response->successful()) {
                $result = $response->json();
                
                // Verificar se a resposta tem o formato esperado
                if (!isset($result['success'])) {
                    // Se não tem 'success', assumir que é válido se tem 'license'
                    if (isset($result['license'])) {
                        $result['success'] = true;
                    } else {
                        $result['success'] = false;
                        $result['message'] = 'Resposta inválida do Master Panel';
                    }
                }
                
                if (config('master-panel.cache_license_validation', true)) {
                    $ttl = config('master-panel.cache_ttl', 60);
                    Cache::put($cacheKey, $result, now()->addMinutes($ttl));
                }
                
                return $result;
            }

            // Se a resposta não foi bem-sucedida, verificar o status
            $status = $response->status();
            $body = $response->json();
            
            $errorMessage = $body['message'] ?? "Erro ao validar licença (Status HTTP: {$status})";
            
            // Adicionar detalhes adicionais baseado no status
            if ($status === 404) {
                $errorMessage .= '. Endpoint não encontrado. Verifique se a URL do Master Panel está correta.';
            } elseif ($status === 403) {
                $errorMessage .= '. Licença inválida, expirada ou bloqueada.';
            } elseif ($status >= 500) {
                $errorMessage .= '. Erro no servidor Master Panel. Tente novamente mais tarde.';
            } else {
                $errorMessage .= '. Verifique a URL do Master Panel (' . $this->baseUrl . ') e sua conexão com a internet.';
            }
            
            return [
                'success' => false,
                'message' => $errorMessage,
                'status' => $status,
                'body' => $body
            ];
        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            Log::error('Master Panel connection error', [
                'url' => $url ?? 'N/A',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return [
                'success' => false,
                'message' => 'Erro de conexão com Master Panel (' . $this->baseUrl . '). Verifique se a URL está correta e se há conexão com a internet. Erro: ' . $e->getMessage()
            ];
        } catch (\Exception $e) {
            Log::error('Master Panel validation error', [
                'url' => $url ?? 'N/A',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return [
                'success' => false,
                'message' => 'Erro ao validar licença: ' . $e->getMessage() . ' (URL: ' . ($url ?? 'N/A') . ')'
            ];
        }
    }

    /**
     * Buscar informações do cliente
     */
    public function getClientInfo()
    {
        try {
            $response = Http::timeout(config('master-panel.api_timeout'))
                ->withHeaders([
                    'X-API-Key' => $this->apiKey,
                    'X-API-Secret' => $this->apiSecret,
                ])
                ->get(rtrim($this->baseUrl, '/') . '/api/v1/client-info');

            if ($response->successful()) {
                return $response->json();
            }

            return null;
        } catch (\Exception $e) {
            Log::error('Master Panel client info error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Verificar atualizações disponíveis
     */
    public function checkUpdates($currentVersion)
    {
        try {
            $url = rtrim($this->baseUrl, '/') . '/api/v1/check-updates';
            $response = Http::timeout(config('master-panel.api_timeout'))
                ->get($url, ['current_version' => $currentVersion]);

            if ($response->successful()) {
                return $response->json();
            }

            return null;
        } catch (\Exception $e) {
            Log::error('Master Panel check updates error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Listar plugins disponíveis
     */
    public function getAvailablePlugins()
    {
        try {
            $response = Http::timeout(config('master-panel.api_timeout'))
                ->withHeaders([
                    'X-API-Key' => $this->apiKey,
                    'X-API-Secret' => $this->apiSecret,
                ])
                ->get(rtrim($this->baseUrl, '/') . '/api/v1/plugins');

            if ($response->successful()) {
                return $response->json();
            }

            return null;
        } catch (\Exception $e) {
            Log::error('Master Panel plugins error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Ativar plugin
     */
    public function activatePlugin($pluginSlug, $licenseKey = null)
    {
        try {
            $response = Http::timeout(config('master-panel.api_timeout'))
                ->withHeaders([
                    'X-API-Key' => $this->apiKey,
                    'X-API-Secret' => $this->apiSecret,
                ])
                ->post(rtrim($this->baseUrl, '/') . '/api/v1/activate-plugin', [
                    'plugin_slug' => $pluginSlug,
                    'license_key' => $licenseKey,
                ]);

            if ($response->successful()) {
                return $response->json();
            }

            return ['success' => false, 'error' => 'Erro ao ativar plugin'];
        } catch (\Exception $e) {
            Log::error('Master Panel activate plugin error: ' . $e->getMessage());
            return ['success' => false, 'error' => 'Erro de conexão'];
        }
    }

    /**
     * Obter novos plugins disponíveis
     */
    public function getNewPlugins()
    {
        try {
            $plugins = $this->getAvailablePlugins();
            
            if (!$plugins || !isset($plugins['plugins'])) {
                return [];
            }

            // Retornar apenas plugins novos (pode implementar lógica de comparação)
            return $plugins['plugins'] ?? [];
        } catch (\Exception $e) {
            Log::error('Master Panel get new plugins error: ' . $e->getMessage());
            return [];
        }
    }
}

