<?php

namespace App\Services;

use App\Models\News;
use App\Models\Category;
use App\Models\AutoPostJob;
use App\Models\AutoPostLog;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class AutoPostService
{
    protected $adapters = [];

    public function __construct()
    {
        $this->loadAdapters();
    }

    /**
     * Carregar adapters disponíveis
     */
    protected function loadAdapters()
    {
        $adapterPath = app_path('Services/AutoPost/Adapters');
        
        if (is_dir($adapterPath)) {
            $files = glob($adapterPath . '/*Adapter.php');
            foreach ($files as $file) {
                $className = 'App\\Services\\AutoPost\\Adapters\\' . basename($file, '.php');
                if (class_exists($className)) {
                    $adapter = new $className();
                    $this->adapters[$adapter->getName()] = $adapter;
                }
            }
        }
    }

    /**
     * Executar job de importação
     */
    public function runJob(AutoPostJob $job)
    {
        $adapter = $this->getAdapter($job->provider_type);
        
        if (!$adapter) {
            throw new \Exception("Adapter não encontrado: {$job->provider_type}");
        }

        try {
            $items = $adapter->fetch($job->provider_url, $job->provider_config ?? []);
            
            $imported = 0;
            $skipped = 0;

            foreach (array_slice($items, 0, $job->max_import ?? 5) as $item) {
                // Verificar duplicidade
                if ($this->isDuplicate($item)) {
                    $skipped++;
                    continue;
                }

                // Criar notícia
                $news = $this->createNewsFromItem($item, $job);
                
                if ($news) {
                    $imported++;
                    
                    // Log
                    AutoPostLog::create([
                        'job_id' => $job->id,
                        'provider' => $job->provider_name,
                        'item_url' => $item['url'] ?? null,
                        'item_title' => $item['title'] ?? 'Sem título',
                        'status' => 'success',
                        'news_id' => $news->id,
                    ]);
                }
            }

            $job->update([
                'last_run_at' => now(),
                'last_imported' => $imported,
                'last_skipped' => $skipped,
            ]);

            return [
                'success' => true,
                'imported' => $imported,
                'skipped' => $skipped,
            ];

        } catch (\Exception $e) {
            Log::error("AutoPost Error: " . $e->getMessage());
            
            AutoPostLog::create([
                'job_id' => $job->id,
                'provider' => $job->provider_name,
                'status' => 'error',
                'error_message' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    /**
     * Verificar se item já foi importado
     */
    protected function isDuplicate($item)
    {
        $url = $item['url'] ?? null;
        $hash = $item['hash'] ?? md5($item['title'] ?? '');

        if ($url) {
            return News::where('source_url', $url)->exists();
        }

        return News::where('source_hash', $hash)->exists();
    }

    /**
     * Criar notícia a partir do item importado
     */
    protected function createNewsFromItem($item, AutoPostJob $job)
    {
        // Mapear categoria
        $categoryId = $this->mapCategory($item, $job);
        
        // Criar notícia
        $news = News::create([
            'title' => $item['title'],
            'subtitle' => $item['subtitle'] ?? null,
            'body' => $item['body'] ?? $item['content'] ?? '',
            'excerpt' => $item['excerpt'] ?? Str::limit(strip_tags($item['body'] ?? $item['content'] ?? ''), 200),
            'slug' => Str::slug($item['title']),
            'category_id' => $categoryId,
            'user_id' => $job->user_id ?? 1,
            'source' => $job->provider_name,
            'source_url' => $item['url'] ?? null,
            'source_hash' => md5($item['url'] ?? $item['title']),
            'published_at' => isset($item['published_at']) ? \Carbon\Carbon::parse($item['published_at']) : now(),
            'status' => $job->auto_publish === 'published' ? 'published' : 'draft',
            'is_featured' => false,
            'is_urgent' => false,
        ]);

        // Upload de imagens se existirem
        if (isset($item['images']) && is_array($item['images'])) {
            $this->downloadImages($news, $item['images']);
        }

        // Notificações se configurado
        if ($job->send_notifications && $news->status === 'published') {
            $this->sendNotifications($news, $job);
        }

        // Publicar em redes sociais se configurado
        if ($job->publish_social && $news->status === 'published') {
            $this->publishToSocial($news, $job);
        }

        return $news;
    }

    /**
     * Mapear categoria
     */
    protected function mapCategory($item, AutoPostJob $job)
    {
        $categoryName = $item['category'] ?? null;
        
        if ($categoryName && $job->create_category_if_not_exists) {
            $category = Category::firstOrCreate(
                ['slug' => Str::slug($categoryName)],
                [
                    'name' => $categoryName,
                    'status' => 'active',
                    'color' => '#667eea',
                ]
            );
            return $category->id;
        }

        return $job->default_category_id ?? Category::first()->id ?? 1;
    }

    /**
     * Download de imagens
     */
    protected function downloadImages($news, $images)
    {
        // Implementar download de imagens
        // Por enquanto, apenas salvar URLs
        $news->update(['images' => json_encode($images)]);
    }

    /**
     * Enviar notificações
     */
    protected function sendNotifications($news, $job)
    {
        // Implementar notificações push e WhatsApp
    }

    /**
     * Publicar em redes sociais
     */
    protected function publishToSocial($news, $job)
    {
        // Implementar publicação automática
    }

    /**
     * Obter adapter
     */
    public function getAdapter($type)
    {
        return $this->adapters[$type] ?? null;
    }

    /**
     * Listar adapters disponíveis
     */
    public function getAvailableAdapters()
    {
        return array_keys($this->adapters);
    }
}

