<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

class Podcast extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'description',
        'audio_url',
        'thumbnail',
        'duration',
        'author',
        'published_at',
        'status',
        'views',
        'downloads'
    ];

    protected $casts = [
        'published_at' => 'datetime',
        'views' => 'integer',
        'downloads' => 'integer'
    ];

    /**
     * Escopo para podcasts publicados
     */
    public function scopePublished($query)
    {
        return $query->where('status', 'published')
            ->where('published_at', '<=', now());
    }

    /**
     * Escopo para podcasts mais recentes
     */
    public function scopeRecent($query, $limit = 6)
    {
        return $query->published()
            ->latest('published_at')
            ->limit($limit);
    }

    /**
     * Obter URL da thumbnail
     */
    public function getThumbnailUrlAttribute()
    {
        if ($this->thumbnail) {
            return Storage::url($this->thumbnail);
        }
        return asset('images/default-podcast.jpg');
    }

    /**
     * Obter duração formatada
     */
    public function getFormattedDurationAttribute()
    {
        if (!$this->duration) {
            return '--:--';
        }
        
        $minutes = floor($this->duration / 60);
        $seconds = $this->duration % 60;
        
        return sprintf('%02d:%02d', $minutes, $seconds);
    }
}

