<?php

namespace App\Http\Controllers;

use App\Models\News;
use App\Models\Category;
use App\Models\Video;
use App\Models\Gallery;
use App\Models\Podcast;
use App\Models\AdCampaign;
use App\Models\AdPosition;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class HomeController extends Controller
{
    /**
     * Home - Template V1 (Moderno)
     */
    public function indexV1()
    {
        // Banner Principal (Hero) - Última notícia em destaque
        $heroNews = News::where('status', 'published')
            ->where('is_featured', true)
            ->with(['category', 'user'])
            ->latest('published_at')
            ->first();

        // 2 chamadas menores ao lado do hero
        $heroSideNews = News::where('status', 'published')
            ->where('id', '!=', $heroNews->id ?? 0)
            ->with(['category', 'user'])
            ->latest('published_at')
            ->limit(2)
            ->get();

        // Bloco de Destaques (3-4 notícias)
        $featuredNews = News::where('status', 'published')
            ->where('is_featured', true)
            ->where('id', '!=', $heroNews->id ?? 0)
            ->with(['category', 'user'])
            ->latest('published_at')
            ->limit(4)
            ->get();

        // Trending / Em alta (mais acessadas hoje)
        $trendingNews = News::where('status', 'published')
            ->whereDate('published_at', today())
            ->orderBy('views', 'desc')
            ->limit(5)
            ->get();

        // Blocos por Categoria
        $categories = Category::where('status', 'active')
            ->orderBy('order')
            ->get();

        $categoryBlocks = [];
        foreach ($categories->take(6) as $category) {
            $featured = News::where('status', 'published')
                ->where('category_id', $category->id)
                ->where('is_headline', true)
                ->with(['category', 'user'])
                ->latest('published_at')
                ->first();
            
            $postsQuery = News::where('status', 'published')
                ->where('category_id', $category->id);
            
            if ($featured) {
                $postsQuery->where('id', '!=', $featured->id);
            }
            
            $categoryBlocks[$category->id] = [
                'category' => $category,
                'featured' => $featured,
                'posts' => $postsQuery->with(['category', 'user'])
                    ->latest('published_at')
                    ->limit(6)
                    ->get(),
            ];
        }

        // Bloco de Vídeos
        $featuredVideo = Video::where('status', 'active')
            ->latest()
            ->first();

        $recentVideos = Video::where('status', 'active')
            ->where('id', '!=', $featuredVideo->id ?? 0)
            ->latest()
            ->limit(6)
            ->get();
            
        // Extrair video_id do embed_code ou url para YouTube/Vimeo
        if ($featuredVideo && $featuredVideo->embed_code) {
            // Tentar extrair ID do YouTube
            if (preg_match('/youtube\.com\/embed\/([a-zA-Z0-9_-]+)/', $featuredVideo->embed_code, $matches)) {
                $featuredVideo->video_id = $matches[1];
            } elseif (preg_match('/youtu\.be\/([a-zA-Z0-9_-]+)/', $featuredVideo->url ?? '', $matches)) {
                $featuredVideo->video_id = $matches[1];
            }
        }

        // Bloco de Colunistas (usuários com mais publicações)
        $columnists = \App\Models\User::whereHas('news', function($query) {
                $query->where('status', 'published');
            })
            ->withCount('news')
            ->orderBy('news_count', 'desc')
            ->limit(6)
            ->get()
            ->map(function($user) {
                $user->latest_news = $user->news()
                    ->where('status', 'published')
                    ->latest('published_at')
                    ->first();
                return $user;
            });

        // Bloco de Podcast (opcional)
        $featuredPodcast = Podcast::where('status', 'published')
            ->latest('published_at')
            ->first();
            
        $recentPodcasts = Podcast::where('status', 'published')
            ->where('id', '!=', $featuredPodcast->id ?? 0)
            ->latest('published_at')
            ->limit(6)
            ->get();

        // Anúncios para sidebar
        $sidebarAds = $this->getSidebarAds();

        return view('frontend.v1.home', compact(
            'heroNews',
            'heroSideNews',
            'featuredNews',
            'trendingNews',
            'categoryBlocks',
            'featuredVideo',
            'recentVideos',
            'columnists',
            'featuredPodcast',
            'recentPodcasts',
            'sidebarAds'
        ));
    }

    /**
     * Home - Template V2 (Blog Simples)
     */
    public function indexV2()
    {
        // Feed linear de notícias
        $news = News::where('status', 'published')
            ->with(['category', 'user'])
            ->latest('published_at')
            ->paginate(10);

        // Mais lidas
        $mostRead = News::where('status', 'published')
            ->orderBy('views', 'desc')
            ->limit(5)
            ->get();

        // Últimos posts para sidebar
        $latestPosts = News::where('status', 'published')
            ->with(['category', 'user'])
            ->latest('published_at')
            ->limit(5)
            ->get();

        // Anúncios para sidebar
        $sidebarAds = $this->getSidebarAds();

        return view('frontend.v2.home', compact(
            'news',
            'mostRead',
            'latestPosts',
            'sidebarAds'
        ));
    }

    /**
     * Obter anúncios para sidebar
     */
    protected function getSidebarAds()
    {
        $ads = [
            'vertical_300x600' => AdCampaign::whereHas('position', function($q) {
                    $q->where('slug', 'arranha-ceu-300x600');
                })
                ->active()
                ->inRandomOrder()
                ->first(),
            'rectangle_300x250' => AdCampaign::whereHas('position', function($q) {
                    $q->where('slug', 'retangulo-300x250');
                })
                ->active()
                ->inRandomOrder()
                ->first(),
        ];

        return $ads;
    }
}

