<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Municipality;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class MunicipalityController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $query = Municipality::query();

        if ($request->filled('search')) {
            $query->where('name', 'like', "%{$request->search}%");
        }

        if ($request->filled('uf')) {
            $query->where('uf', $request->uf);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $municipalities = $query->orderBy('uf')->orderBy('name')->paginate(50);
        $states = Municipality::select('uf')->distinct()->orderBy('uf')->pluck('uf');

        return view('admin.municipalities.index', compact('municipalities', 'states'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'uf' => 'required|string|size:2',
            'ibge_code' => 'nullable|string|max:10',
            'status' => 'required|in:active,inactive',
        ]);

        $validated['slug'] = Str::slug($validated['name']);

        Municipality::create($validated);

        return redirect()->route('admin.municipalities.index')
            ->with('success', 'Município cadastrado com sucesso!');
    }

    public function update(Request $request, Municipality $municipality)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'uf' => 'required|string|size:2',
            'ibge_code' => 'nullable|string|max:10',
            'status' => 'required|in:active,inactive',
        ]);

        if ($municipality->name !== $validated['name']) {
            $validated['slug'] = Str::slug($validated['name']);
        }

        $municipality->update($validated);

        return redirect()->route('admin.municipalities.index')
            ->with('success', 'Município atualizado com sucesso!');
    }

    public function destroy(Municipality $municipality)
    {
        // Verificar se tem notícias vinculadas
        if ($municipality->news()->count() > 0) {
            return back()->with('error', 'Não é possível excluir município com notícias vinculadas!');
        }

        $municipality->delete();

        return redirect()->route('admin.municipalities.index')
            ->with('success', 'Município excluído com sucesso!');
    }

    public function import()
    {
        // Método para importar municípios em massa (IBGE)
        // Implementar importação de CSV ou API IBGE
        return view('admin.municipalities.import');
    }
}

