<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AdCampaign;
use App\Models\AdPosition;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class AdController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $query = AdCampaign::with(['user', 'position']);

        if ($request->filled('search')) {
            $query->where('name', 'like', "%{$request->search}%");
        }

        if ($request->filled('position_id')) {
            $query->where('position_id', $request->position_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $campaigns = $query->orderBy('created_at', 'desc')->paginate(20);
        $positions = AdPosition::active()->get();

        return view('admin.ads.index', compact('campaigns', 'positions'));
    }

    public function create()
    {
        $positions = AdPosition::active()->get();
        return view('admin.ads.create', compact('positions'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'position_id' => 'required|exists:ad_positions,id',
            'name' => 'required|string|max:255',
            'format' => 'required|in:image,html,script',
            'content' => 'required|string',
            'image' => 'nullable|image|max:2048',
            'link' => 'nullable|url|max:500',
            'target' => 'required|in:all,home,pages,municipality,blog,column',
            'target_pages' => 'nullable|array',
            'target_municipality_id' => 'nullable|exists:municipalities,id',
            'target_blog_id' => 'nullable|integer',
            'target_column' => 'nullable|string',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
            'frequency' => 'nullable|integer|min:1',
            'rotations' => 'nullable|integer|min:1',
            'impressions_limit' => 'nullable|integer|min:0',
            'clicks_limit' => 'nullable|integer|min:0',
            'status' => 'required|in:active,paused,finished',
        ]);

        $validated['user_id'] = Auth::id();

        if ($request->hasFile('image')) {
            $validated['image'] = $request->file('image')->store('ads', 'public');
        }

        AdCampaign::create($validated);

        return redirect()->route('admin.ads.index')
            ->with('success', 'Campanha criada com sucesso!');
    }

    public function edit(AdCampaign $ad)
    {
        $positions = AdPosition::active()->get();
        return view('admin.ads.edit', compact('ad', 'positions'));
    }

    public function update(Request $request, AdCampaign $ad)
    {
        $validated = $request->validate([
            'position_id' => 'required|exists:ad_positions,id',
            'name' => 'required|string|max:255',
            'format' => 'required|in:image,html,script',
            'content' => 'required|string',
            'image' => 'nullable|image|max:2048',
            'link' => 'nullable|url|max:500',
            'target' => 'required|in:all,home,pages',
            'target_pages' => 'nullable|array',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
            'impressions_limit' => 'nullable|integer|min:0',
            'clicks_limit' => 'nullable|integer|min:0',
            'status' => 'required|in:active,paused,finished',
        ]);

        if ($request->hasFile('image')) {
            if ($ad->image) {
                Storage::disk('public')->delete($ad->image);
            }
            $validated['image'] = $request->file('image')->store('ads', 'public');
        }

        $ad->update($validated);

        return redirect()->route('admin.ads.index')
            ->with('success', 'Campanha atualizada com sucesso!');
    }

    public function destroy(AdCampaign $ad)
    {
        if ($ad->image) {
            Storage::disk('public')->delete($ad->image);
        }
        $ad->delete();

        return redirect()->route('admin.ads.index')
            ->with('success', 'Campanha excluída com sucesso!');
    }

    // Gerenciar posições
    public function positions()
    {
        $positions = AdPosition::orderBy('name')->get();
        return view('admin.ads.positions', compact('positions'));
    }

    public function showPosition(AdPosition $position)
    {
        return response()->json($position);
    }

    public function storePosition(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'width' => 'required|integer|min:1',
            'height' => 'required|integer|min:1',
            'description' => 'nullable|string',
            'status' => 'required|in:active,inactive',
        ]);

        $validated['slug'] = \Illuminate\Support\Str::slug($validated['name']);

        AdPosition::create($validated);

        return redirect()->route('admin.ads.positions')
            ->with('success', 'Posição criada com sucesso!');
    }

    public function updatePosition(Request $request, AdPosition $position)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'width' => 'required|integer|min:1',
            'height' => 'required|integer|min:1',
            'description' => 'nullable|string',
            'status' => 'required|in:active,inactive',
        ]);

        if ($position->name !== $validated['name']) {
            $validated['slug'] = \Illuminate\Support\Str::slug($validated['name']);
        }

        $position->update($validated);

        if ($request->expectsJson()) {
            return response()->json(['success' => true]);
        }

        return redirect()->route('admin.ads.positions')
            ->with('success', 'Posição atualizada com sucesso!');
    }

    public function destroyPosition(AdPosition $position)
    {
        if ($position->campaigns()->count() > 0) {
            if (request()->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Não é possível excluir posição com campanhas vinculadas!'
                ], 400);
            }
            return redirect()->route('admin.ads.positions')
                ->with('error', 'Não é possível excluir posição com campanhas vinculadas!');
        }

        $position->delete();

        if (request()->expectsJson()) {
            return response()->json(['success' => true]);
        }

        return redirect()->route('admin.ads.positions')
            ->with('success', 'Posição excluída com sucesso!');
    }

    public function createDefaultPositions()
    {
        $defaultPositions = [
            ['name' => 'Banner Home 970x150', 'width' => 970, 'height' => 150, 'slug' => 'banner-home-970x150', 'description' => 'Banner principal na home'],
            ['name' => 'Popup 580x400', 'width' => 580, 'height' => 400, 'slug' => 'popup-580x400', 'description' => 'Popup modal'],
            ['name' => 'Arranha-céu 300x600', 'width' => 300, 'height' => 600, 'slug' => 'arranha-ceu-300x600', 'description' => 'Banner vertical'],
            ['name' => 'Retângulo 300x250', 'width' => 300, 'height' => 250, 'slug' => 'retangulo-300x250', 'description' => 'Banner retângulo médio'],
            ['name' => 'Super Banner 970x250', 'width' => 970, 'height' => 250, 'slug' => 'super-banner-970x250', 'description' => 'Banner super horizontal'],
            ['name' => 'Intersticial', 'width' => 320, 'height' => 480, 'slug' => 'intersticial', 'description' => 'Anúncio intersticial mobile'],
            ['name' => 'Mobile Sticky', 'width' => 320, 'height' => 50, 'slug' => 'mobile-sticky', 'description' => 'Banner sticky mobile'],
            ['name' => 'In-Article', 'width' => 300, 'height' => 250, 'slug' => 'in-article', 'description' => 'Inserção entre parágrafos'],
        ];

        foreach ($defaultPositions as $pos) {
            AdPosition::firstOrCreate(
                ['slug' => $pos['slug']],
                array_merge($pos, ['status' => 'active'])
            );
        }

        return response()->json(['success' => true, 'message' => 'Posições padrão criadas com sucesso!']);
    }
}

