<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Post;
use App\Models\Category;
use App\Models\Ad;
use App\Models\Video;
use App\Models\Setting;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    public function index()
    {
        // Verificar qual template está ativo
        $template = Setting::get('active_template', 'v1');
        
        if ($template === 'v2') {
            return $this->indexV2();
        }
        
        return $this->indexV1();
    }

    public function indexV1()
    {
        // Destaques (slide)
        $slides = Post::where('status', 'published')
            ->where('is_slide', true)
            ->whereNotNull('published_at')
            ->where('published_at', '<=', now())
            ->orderBy('published_at', 'desc')
            ->limit(5)
            ->get();

        // Manchete principal
        $manchete = Post::where('status', 'published')
            ->where('is_manchete', true)
            ->whereNotNull('published_at')
            ->where('published_at', '<=', now())
            ->orderBy('published_at', 'desc')
            ->first();

        // Destaques
        $featured = Post::where('status', 'published')
            ->where('featured', true)
            ->whereNotNull('published_at')
            ->where('published_at', '<=', now())
            ->orderBy('published_at', 'desc')
            ->limit(4)
            ->get();

        // Últimas notícias
        $latest = Post::where('status', 'published')
            ->whereNotNull('published_at')
            ->where('published_at', '<=', now())
            ->orderBy('published_at', 'desc')
            ->limit(12)
            ->get();

        // Notícias por categoria
        $categories = Category::where('active', true)
            ->orderBy('order')
            ->limit(6)
            ->get()
            ->map(function($category) {
                $category->posts = Post::where('category_id', $category->id)
                    ->where('status', 'published')
                    ->whereNotNull('published_at')
                    ->where('published_at', '<=', now())
                    ->orderBy('published_at', 'desc')
                    ->limit(6)
                    ->get();
                return $category;
            });

        // Vídeos
        $videos = Video::where('status', 'published')
            ->where('publish_on_site', true)
            ->whereNotNull('published_at')
            ->where('published_at', '<=', now())
            ->orderBy('published_at', 'desc')
            ->limit(6)
            ->get();

        // Anúncios
        $ads = $this->getAds();

        return view('frontend.v1.home', compact(
            'slides',
            'manchete',
            'featured',
            'latest',
            'categories',
            'videos',
            'ads'
        ));
    }

    public function indexV2()
    {
        // Feed linear simples
        $posts = Post::where('status', 'published')
            ->whereNotNull('published_at')
            ->where('published_at', '<=', now())
            ->orderBy('published_at', 'desc')
            ->paginate(10);

        // Anúncios para sidebar
        $ads = $this->getAds();

        return view('frontend.v2.home', compact('posts', 'ads'));
    }

    private function getAds()
    {
        return Ad::where('status', 'active')
            ->where(function($query) {
                $query->whereNull('start_at')
                      ->orWhere('start_at', '<=', now());
            })
            ->where(function($query) {
                $query->whereNull('end_at')
                      ->orWhere('end_at', '>=', now());
            })
            ->where(function($query) {
                $query->where('target', 'all')
                      ->orWhere('target', 'home');
            })
            ->orderBy('created_at', 'desc')
            ->get();
    }
}

